<?php
/**
 * 2007-2025 Shipday
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    Shipday <support@shipday.com>
 * @copyright 2007-2025 Shipday
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class OrderEvent
{
    public $event;
    public $payload;

    public function __construct($event, $payload)
    {
        $this->event = $event;
        $this->payload = $payload;
    }
}

class ShipdayIntegrationModule extends Module
{
    const BASE_WEBHOOK_URL = 'https://integration.shipday.com';

    public function __construct()
    {
        $this->name = 'shipdayintegrationmodule';
        $this->tab = 'shipping_logistics';
        $this->version = '1.0.0';
        $this->author = 'Shipday';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.7.0.0',
            'max' => _PS_VERSION_,
        ];

        parent::__construct();

        $this->displayName = $this->l('Shipday Integration');
        $this->description = $this->l('Sends order details to Shipday when an order is created or updated.');

        if (version_compare(_PS_VERSION_, '1.7.7', '>=')) {
            $this->context->smarty->registerPlugin('modifier', 'safe_stripslashes', 'stripslashes');
            $this->context->smarty->registerPlugin('modifier', 'safe_md5', 'md5');
        }

        if (PHP_VERSION_ID >= 80100) {
            set_error_handler(function ($errno, $errstr) {
                return strpos($errstr, 'strtolower(): Passing null') !== false;
            }, E_DEPRECATED);
        }
    }

    public function install()
    {
        return parent::install()
            && $this->registerHook('actionOrderStatusPostUpdate')
            && $this->registerHook('actionValidateOrder')
            && $this->registerHook('actionObjectOrderUpdateAfter')
            && $this->installConfiguration();
    }

    public function uninstall()
    {
        return parent::uninstall();
    }

    public function installConfiguration()
    {
        if (!Configuration::get('ORDER_WEBHOOK_API_KEY')) {
            Configuration::updateValue('ORDER_WEBHOOK_API_KEY', 'default-api-key');
        }
        return true;
    }

    public function getContent()
    {
        if (Tools::isSubmit('submit_shipdayintegrationmodule')) {
            Configuration::updateValue('ORDER_WEBHOOK_API_KEY', Tools::getValue('ORDER_WEBHOOK_API_KEY'));
        }

        $this->context->smarty->assign([
            'ORDER_WEBHOOK_API_KEY' => Configuration::get('ORDER_WEBHOOK_API_KEY'),
        ]);

        return $this->display(__FILE__, 'views/templates/admin/configure.tpl');
    }

    public function hookActionValidateOrder($params)
    {
        $order = $params['order'];
        $endpoint = '/integration/prestashop/event/order/create';
        $fullUrl = rtrim(self::BASE_WEBHOOK_URL, '/') . $endpoint;

        $this->sendShipdayIntegrationModulePayload($fullUrl, $order, 'order.created');
    }

    public function hookActionObjectOrderUpdateAfter($params)
    {
        $order = null;
        if (isset($params['object']) && $params['object'] instanceof Order) {
            $order = $params['object'];
        } elseif (isset($params['payload']['id'])) {
            $order = new Order((int) $params['payload']['id']);
        }

        if (!$order || !Validate::isLoadedObject($order)) {
            PrestaShopLogger::addLog('Invalid order in hookActionObjectOrderUpdateAfter', 3);
            return;
        }

        $endpoint = '/integration/prestashop/event/order/update';
        $fullUrl = rtrim(self::BASE_WEBHOOK_URL, '/') . $endpoint;

        $this->sendShipdayIntegrationModulePayload($fullUrl, $order, 'order.updated');
    }

    public function hookActionOrderStatusPostUpdate($params)
    {
        if (!isset($params['id_order']) || !isset($params['newOrderStatus'])) {
            PrestaShopLogger::addLog('Missing parameters in hookActionOrderStatusPostUpdate', 3);
            return;
        }

        $orderId = (int) $params['id_order'];
        $newStatus = $params['newOrderStatus'];
        $canceledStatusId = (int) Configuration::get('PS_OS_CANCELED');

        if ((int) $newStatus->id === $canceledStatusId) {
            $order = new Order($orderId);
            if (!Validate::isLoadedObject($order)) {
                PrestaShopLogger::addLog('Invalid order #' . $orderId . ' in hookActionOrderStatusPostUpdate', 3);
                return;
            }

            $payload = [
                'event' => 'order.canceled',
                'payload' => [
                    'order_id' => $order->id,
                    'order_ref' => $order->reference,
                    'status' => $newStatus->name,
                    'canceled_at' => date('Y-m-d H:i:s'),
                ],
            ];

            $endpoint = '/integration/prestashop/event/order/cancel';
            $fullUrl = rtrim(self::BASE_WEBHOOK_URL, '/') . $endpoint;
            $apiKey = Configuration::get('ORDER_WEBHOOK_API_KEY');

            $this->sendWebhook($fullUrl, json_encode($payload), $apiKey);
        }
    }

    private function sendShipdayIntegrationModulePayload($fullUrl, Order $order, $eventName)
    {
        $deliveryAddress = new Address($order->id_address_delivery);
        $customer = new Customer($order->id_customer);
        $orderItems = $order->getProducts();

        $stateName = '';
        if ($deliveryAddress->id_state) {
            $state = new State($deliveryAddress->id_state);
            $stateName = $state->name;
        }

        $totalTax = $order->total_paid_tax_incl - $order->total_paid_tax_excl;

        $orderFields = $order->getFields();
        $orderFields['total_tax'] = round($totalTax, 2);

        $orderMessage = '';
        $messages = Message::getMessagesByOrderId($order->id);
        if (!empty($messages)) {
            $orderMessage = implode("\n", array_column($messages, 'message'));
        }

        $storeData = [];
        $stores = Store::getStores((int) Context::getContext()->language->id);
        if (!empty($stores)) {
            $firstStore = $stores[0];
            $storeData = [
                'id' => $firstStore['id_store'],
                'name' => $firstStore['name'],
                'address' => $firstStore['address1'],
                'city' => $firstStore['city'],
                'postcode' => $firstStore['postcode'],
                'state' => State::getNameById($firstStore['id_state']),
                'country' => Country::getNameById((int) Context::getContext()->language->id, $firstStore['id_country']),
                'phone' => $firstStore['phone'],
            ];
        }

        $payload = [
            'event' => $eventName,
            'payload' => [
                'order' => $orderFields,
                'delivery_address' => array_merge(
                    $deliveryAddress->getFields(),
                    ['state_name' => $stateName]
                ),
                'customer' => $customer->getFields(),
                'order_items' => $orderItems,
                'order_message' => $orderMessage,
                'store' => $storeData,
            ],
        ];

        $apiKey = Configuration::get('ORDER_WEBHOOK_API_KEY');
        $this->sendWebhook($fullUrl, json_encode($payload), $apiKey);
    }

    private function sendWebhook($url, $data, $apiKey)
    {
        if (!is_string($data)) {
            $data = json_encode($data);
        }

        if (json_last_error() !== JSON_ERROR_NONE) {
            PrestaShopLogger::addLog('JSON encoding error: ' . json_last_error_msg(), 3);
            return;
        }

        $ch = curl_init($url);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'api-key: ' . $apiKey,
            'presta-code: X7K2M9PLWQ4ZBJY3N8TC',
            'Content-Length: ' . strlen($data),
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_VERBOSE, true);

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            PrestaShopLogger::addLog('cURL Error: ' . curl_error($ch), 3);
        } else {
            PrestaShopLogger::addLog('Webhook Response: ' . $response, 1);
        }

        curl_close($ch);
    }

    public function upgradeModule($moduleVersion)
    {
        $updateFolder = __DIR__ . '/update/';
        $updates = [
            '1.0.0' => 'update-1.0.0.php',
        ];

        foreach ($updates as $version => $file) {
            if (version_compare($moduleVersion, $version, '<')) {
                $filePath = $updateFolder . $file;

                if (file_exists($filePath)) {
                    require_once $filePath;
                    $updateFunction = 'update_' . str_replace('.', '_', $version);
                    if (function_exists($updateFunction)) {
                        if (!$updateFunction()) {
                            return false;
                        }
                    }
                }
            }
        }
        return true;
    }
}
